/******************************************************************************/
/*                                                                            */
/*  Title       : lcw                                                         */
/*  Date        : 27JUN99                                                     */
/*  Last change : 15AUG99                                                     */
/*                                                                            */
/*  Copyright (C) 1999 Roland Maetche  DD5RM                                  */
/*                                                                            */
/* This program is free software; you can redistribute it and/or modify       */
/* it under the terms of the GNU General Public License as published by       */
/* the Free Software Foundation; either version 2 of the License, or          */
/* (at your option) any later version.                                        */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA  */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* Includes                                                                   */
/******************************************************************************/
#include <curses.h>
#include <stdlib.h>
#include <sched.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <malloc.h>

#include "menu.h"

/******************************************************************************/
/* Definitions                                                                */
/******************************************************************************/

/******** Compile-time options ********/
/* OPT_CALIBRATE is only used to calibarate			*/
/*	cw-speed (some compiler-warnings will appear)	*/
/* #define	OPT_CALIBRATE							*/

/******** Sound device ********/
#define SOUND_DEVICE	"/dev/dsp"
#define MIN_BLOCKS	2
#ifdef	OPT_CALIBRATE
 #define WAVEBUFS	3		/* to reduce time-error while calculating speed */
#else
 #define WAVEBUFS	6		/* min. 3  to avoid noise ! */
#endif
#define WAVEBUFLEN	4000

/******** Wave generation ********/
#define PI		3.141592654
#define TWOPI		6.283185307
#define HALFPI		1.570796327

/******** CW params ********/
#define	SIGNS_PER_WORD	5.0		/* determined using OPT_CALIBRATE		*/
#define STEPS_PER_SIGN	10.0	/* includes sign- and word-spaces		*/
						/*  (determined using OPT_CALIBRATE)	*/
#define RATIO		3
#define	N_SIGNS		43
/******** codes used in sign-definition ********/
#define	C_END		0
#define	C_DI		1
#define C_DA		2

/******** RC specific ********/
#define ENV_HOME	"HOME"		/* env-value for home-directory (for rc-file) */
#define RCFILE		".lcwrc"
#define RCKEY_FRQ	"freq"
#define RCKEY_AMP	"amp"
#define RCKEY_WPM	"wpm"
#define RCKEY_PSE	"pse"
#define RCKEY_KEY	"key"
#define RCKEY_GRP	"groups"
#define RCKEY_SIGN	"sign"

/******** Parameter ranges ********/
#define	MIN_FRQ		50
#define	MAX_FRQ		2000
#define	DEF_FRQ		600
#define	MIN_AMP		2
#define	MAX_AMP		100
#define	DEF_AMP		50
#define	MIN_WPM		2
#define	MAX_WPM		30
#define	DEF_WPM		12
#define	MIN_PSE		0
#define	MAX_PSE		500
#define	DEF_PSE		0
#define	MIN_KEY		0
#define	MAX_KEY		50
#define	DEF_KEY		5
#define MIN_GRP		1
#define MAX_GRP		99
#define DEF_GRP		10

/******** generation of random signs ********/
#define RNDMOD		3		/* higher values will cause persitance and 	 */
					/*  count to have more effect selecting signs  */

/******** number of lessons ********/
#define N_LESSONS	20

/******** screen, keyboard and window stuff (curses) ********/
#define	KB_ESC		27
#define	SCR_MINX	80		/* minumum screen size to accept */
#define SCR_MINY	24

/*** text for main-menu ***/
#ifdef L_GERMAN
 #define MMENU_TEXT1	"Optionen"
 #define MMENU_KEY1		'O'
 #define MMENU_TEXT2	"Zeichen"
 #define MMENU_KEY2		'Z'
 #define MMENU_TEXT3	"Start"
 #define MMENU_KEY3		'S'
 #define MMENU_TEXT4	"Ende"
 #define MMENU_KEY4		'E'
#else
 #define MMENU_TEXT1	"Options"
 #define MMENU_KEY1		'O'
 #define MMENU_TEXT2	"Signs"
 #define MMENU_KEY2		'S'
 #define MMENU_TEXT3	"Go"
 #define MMENU_KEY3		'G'
 #define MMENU_TEXT4	"Quit"
 #define MMENU_KEY4		'Q'
#endif

/*** text for options-menu ***/
#ifdef L_GERMAN
 #define OMENU_TEXT1	"Frequenz  (Hz)"
 #define OMENU_KEY1		'F'
 #define OMENU_TEXT2	"Amplitude  (%)"
 #define OMENU_KEY2		'A'
 #define OMENU_TEXT3	"Geschw.  (WpM)"
 #define OMENU_KEY3		'G'
 #define OMENU_TEXT4	"Pause      (%)"
 #define OMENU_KEY4		'P'
 #define OMENU_TEXT5	"Tastung       "
 #define OMENU_KEY5		'T'
 #define OMENU_TEXT6	"Gruppen       "
 #define OMENU_KEY6		'G'
 #define OMENU_TEXT7	"Schliessen"
 #define OMENU_KEY7		'S'
#else
 #define OMENU_TEXT1	"Frequency (Hz)"
 #define OMENU_KEY1		'F'
 #define OMENU_TEXT2	"Amplitude  (%)"
 #define OMENU_KEY2		'A'
 #define OMENU_TEXT3	"Speed    (WpM)"
 #define OMENU_KEY3		'S'
 #define OMENU_TEXT4	"Pause      (%)"
 #define OMENU_KEY4		'P'
 #define OMENU_TEXT5	"Keying        "
 #define OMENU_KEY5		'K'
 #define OMENU_TEXT6	"Groups        "
 #define OMENU_KEY6		'G'
 #define OMENU_TEXT7	"Close"
 #define OMENU_KEY7		'C'
#endif

/*** text for signs-menu ***/
#ifdef L_GERMAN
 #define SMENU_TEXT1	"Aktivierung"
 #define SMENU_KEY1		'A'
 #define SMENU_TEXT2	"Lektionen"
 #define SMENU_KEY2		'L'
 #define SMENU_TEXT3	"Wahrscheinl.(%)"
 #define SMENU_KEY3		'W'
 #define SMENU_TEXT4	"Zaehler -> 0"
 #define SMENU_KEY4		'Z'
 #define SMENU_TEXT5	"Schliessen"
 #define SMENU_KEY5		'S'
 #define SMENU_TEXT6	"<ESC><ESC> schliesst Fenster"
#else
 #define SMENU_TEXT1	"Activation"
 #define SMENU_KEY1		'A'
 #define SMENU_TEXT2	"Lessons"
 #define SMENU_KEY2		'L'
 #define SMENU_TEXT3	"Persistance (%)"
 #define SMENU_KEY3		'P'
 #define SMENU_TEXT4	"Reset Counters"
 #define SMENU_KEY4		'R'
 #define SMENU_TEXT5	"Close"
 #define SMENU_KEY5		'C'
 #define SMENU_TEXT6	"Press <ESC><ESC> to close"
#endif

/******************************************************************************/
/* Structures                                                                 */
/******************************************************************************/

/******** cw-sign with some additional data ********/
typedef struct _cwsign {
	unsigned int	flags;
	int			count;
	int			persistance;	
	char			sign;
	unsigned char	code[ 10 ];
} cwsign;


/******** screen and window stuff (curses) ********/

/******** description of value to modify *******/
typedef struct _vmoddat {
	int	winx;
	int	winy;
	char *	text;
	char *	format;
	int *	value;
	int	max;
	int	min;
	int	sav;
} VModDat;

/******************************************************************************/
/* Global variables */
/******************************************************************************/

/******** Params from rc-file ********/
int		Freq =			{ DEF_FRQ };
int		Ampl =			{ DEF_AMP };
int		WpM =			{ DEF_WPM };
int		Pause =			{ DEF_PSE };
int		Ramp =			{ DEF_KEY };
int		Groups =		{ DEF_GRP };

/******** Internal values ********/
int		SamplesPerStep;
double		PhiInc;
#ifdef L_GERMAN
char		ApplText[] =		{ "LCW V0.2.0 von DD5RM" };
#else
char		ApplText[] =		{ "LCW V0.2.0 by DD5RM" };
#endif
int		SampleRate;
int		Fragments;
int		FragSize;
int		SoundDev = { -1 };
int		ValuesInBuffer = { 0 };
int		BufferInUse = { 0 };

/******** Wave-buffers (needed ??) ********/
unsigned char *	WaveBufs[ WAVEBUFS ];
unsigned char * WaveBuf;

/******** CW-signs and more... ********/
cwsign	signs[ N_SIGNS ] = {
	1, 0, 50, 'A',  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'B',  C_DA,  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'C',  C_DA,  C_DI,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'D',  C_DA,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'E',  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'F',  C_DI,  C_DI,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'G',  C_DA,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'H',  C_DI,  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'I',  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'J',  C_DI,  C_DA,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'K',  C_DA,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'L',  C_DI,  C_DA,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'M',  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'N',  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'O',  C_DA,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'P',  C_DI,  C_DA,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'Q',  C_DA,  C_DA,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'R',  C_DI,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'S',  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'T',  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'U',  C_DI,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'V',  C_DI,  C_DI,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'W',  C_DI,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'X',  C_DA,  C_DI,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'Y',  C_DA,  C_DI,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, 'Z',  C_DA,  C_DA,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '1',  C_DI,  C_DA,  C_DA,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '2',  C_DI,  C_DI,  C_DA,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '3',  C_DI,  C_DI,  C_DI,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '4',  C_DI,  C_DI,  C_DI,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '5',  C_DI,  C_DI,  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '6',  C_DA,  C_DI,  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '7',  C_DA,  C_DA,  C_DI,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '8',  C_DA,  C_DA,  C_DA,  C_DI,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '9',  C_DA,  C_DA,  C_DA,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '0',  C_DA,  C_DA,  C_DA,  C_DA,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '.',  C_DI,  C_DA,  C_DI,  C_DA,  C_DI,  C_DA,  C_END, C_END, C_END, C_END,
	1, 0, 50, ',',  C_DA,  C_DA,  C_DI,  C_DI,  C_DA,  C_DA,  C_END, C_END, C_END, C_END,
	1, 0, 50, '?',  C_DI,  C_DI,  C_DA,  C_DA,  C_DI,  C_DI,  C_END, C_END, C_END, C_END,
	1, 0, 50, '=',  C_DA,  C_DI,  C_DI,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '/',  C_DA,  C_DI,  C_DI,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '>',  C_DA,  C_DI,  C_DA,  C_DI,  C_DA,  C_END, C_END, C_END, C_END, C_END,
	1, 0, 50, '+',  C_DI,  C_DA,  C_DI,  C_DA,  C_DI,  C_END, C_END, C_END, C_END, C_END
}; 

/*** signs for lessons ***/
char	LessChars[]		= { "ELV0ASQT2OC=D5R/?I9GXF4NJU7H,KB3PMYZW618." };


/******** screen and window stuff (curses) ********/
WINDOW *	Screen = 	{ NULL };
WINDOW *	TopLine =	{ NULL };
WINDOW *	TextWnd =	{ NULL };
Menu *		MainMenu =	{ NULL };
Menu *		OptMenu =	{ NULL };
Menu *		ModVMenu =	{ NULL };
Menu *		SigMenu =	{ NULL };
Menu *		SigLMenu =	{ NULL };
Menu *		LessMenu =	{ NULL };

/*** text for options-menu ***/
char	OMenuText1[]	= { OMENU_TEXT1 };
char	OMenuText2[]	= { OMENU_TEXT2 };
char	OMenuText3[]	= { OMENU_TEXT3 };
char	OMenuText4[]	= { OMENU_TEXT4 };
char	OMenuText5[]	= { OMENU_TEXT5 };
char	OMenuText6[]	= { OMENU_TEXT6 };

/*** text for signs-menu ***/
char	SMenuText1[]	= { SMENU_TEXT1 };
char	SMenuText2[]	= { SMENU_TEXT2 };
char	SMenuText3[]	= { SMENU_TEXT3 };
char	SMenuText4[]	= { SMENU_TEXT4 };
char	SMenuText5[]	= { SMENU_TEXT5 };
char	SMenuText6[]	= { SMENU_TEXT6 };


/******************************************************************************/
/* Functions */
/******************************************************************************/

/*** Read line from rc-file ***/
int readLine( int fd, char * buffer, int buflen )
{
	int	nread;
	
	nread = 0;
	if( fd != -1 )
	{
		while( read( fd, buffer, 1 ) == 1 )
		{
			if( *buffer == '\n' )
			{
				*buffer = '\0';
				return nread;
			}
			if( nread < buflen )
			{
				buffer++;
				nread++;
			}
		}
		*buffer = '\0';
	}
	return nread;
}

/*** Write current params to rc-file ***/
void write_rc( void )
{
	int	fd;
	char	lbuf[ 100 ];
	int	i;

	lbuf[0] = '\0';
	strcpy( lbuf, getenv( ENV_HOME ) );	/* get path of homedir */
	i = strlen( lbuf );
	lbuf[i++] = '/';
	strcpy( &lbuf[i], RCFILE );
	remove( lbuf );
	fd = open( lbuf, O_CREAT | O_WRONLY, S_IRUSR | S_IWUSR );
	if( fd != -1 )
	{
		sprintf( lbuf, "%s %d\n", RCKEY_FRQ, Freq );
		write( fd, lbuf, strlen(lbuf) );
		sprintf( lbuf, "%s %d\n", RCKEY_AMP, Ampl );
		write( fd, lbuf, strlen(lbuf) );
		sprintf( lbuf, "%s %d\n", RCKEY_WPM, WpM );
		write( fd, lbuf, strlen(lbuf) );
		sprintf( lbuf, "%s %d\n", RCKEY_PSE, Pause );
		write( fd, lbuf, strlen(lbuf) );
		sprintf( lbuf, "%s %d\n", RCKEY_KEY, Ramp );
		write( fd, lbuf, strlen(lbuf) );
		sprintf( lbuf, "%s %d\n", RCKEY_GRP, Groups );
		write( fd, lbuf, strlen(lbuf) );
		for( i=0; i<N_SIGNS; i++ )
		{
			sprintf( lbuf, "%s %c %d %d %d\n", RCKEY_SIGN, signs[i].sign, signs[i].flags, signs[i].count, signs[i].persistance);
			write( fd, lbuf, strlen(lbuf) );
		}
		close(fd);
	}
}

/*** Read rc-file to current params ***/
/***  (if not present, create it using default-params) ***/
void read_rc( void )
{
	int	fd;
	char	lbuf[ 100 ];
	char *	data;
	int	i;
	int	j;
	int	k;
	int	signdata[3];

	lbuf[0] = '\0';
	strcpy( lbuf, getenv( ENV_HOME ) );	/* get path of homedir */
	i = strlen( lbuf );
	lbuf[i++] = '/';
	strcpy( &lbuf[i], RCFILE );
	fd = open( lbuf, O_RDONLY );
	if( fd != -1 )
	{
		while( readLine( fd, lbuf, 100 ) > 0 )
		{
			if( lbuf[0] != '#' )
			{
				data = lbuf;
				while( ( *data != '\0' ) && ( *data != ' ' ) && ( *data != '\t' ) ) data++;
				if( *data != '\0' )
				{
					*data++ = '\0';	/* end of key */
					while( ( *data != '\0' ) && ( ( *data == ' ' ) || ( *data == '\t' ) ) ) data++;
					if( strcmp( lbuf, RCKEY_FRQ ) == 0 )
					{
						Freq = atoi( data );
						if( ( Freq < MIN_FRQ ) || ( Freq > MAX_FRQ ) ) Freq = DEF_FRQ;
					}
					if( strcmp( lbuf, RCKEY_AMP ) == 0 )
					{
						Ampl = atoi( data );
						if( ( Ampl < MIN_AMP ) || ( Ampl > MAX_AMP ) ) Ampl = DEF_AMP;
					}
					if( strcmp( lbuf, RCKEY_WPM ) == 0 )
					{
						WpM = atoi( data );
						if( ( WpM < MIN_WPM ) || ( WpM > MAX_WPM ) ) WpM = DEF_WPM;
					}
					if( strcmp( lbuf, RCKEY_PSE ) == 0 )
					{
						Pause = atoi( data );
						if( ( Pause < MIN_PSE ) || ( Pause > MAX_PSE ) ) Pause = DEF_PSE;
					}
					if( strcmp( lbuf, RCKEY_KEY ) == 0 )
					{
						Ramp = atoi( data );
						if( ( Ramp < MIN_KEY ) || ( Ramp > MAX_KEY ) ) Ramp = DEF_KEY;
					}
					if( strcmp( lbuf, RCKEY_GRP ) == 0 )
					{
						Groups = atoi( data );
						if( ( Groups < MIN_GRP ) || ( Groups > MAX_GRP ) ) Groups = DEF_GRP;
					}
					if( strncmp( lbuf, RCKEY_SIGN ) == 0 )
					{
						j = -1;
						for( i=0; ( ( i<N_SIGNS ) && ( j<0 ) ); i++ )
						{
							if( toupper(*data) == signs[ i ].sign ) j = i;
						}
						if( j >= 0 )
						{
							data++;
							for( k=0; k<3; k++ )
							{
								if( *data != '\0' )
								{
									if( ( *data == ' ' ) || ( *data == '\t' ) )
									{
										while( ( *data != '\0' ) && ( ( *data == ' ' ) || ( *data == '\t' ) ) ) data++;
										if( *data != '\0' )
										{
											signdata[k] = atoi( data );
											while( ( *data != '\0' ) && ( *data != ' ' ) && ( *data != '\t' ) ) data++;
										}
									}
									else signdata[2] = -1;
								}
								else signdata[2] = -1;
							}
							if( signdata[2] != -1 )
							{
								signs[j].flags = (unsigned int) signdata[0];
								signs[j].count = signdata[1];
								signs[j].persistance = signdata[2];
								if( ( signs[j].persistance < 1 ) || ( signs[j].persistance > 100 ) ) signs[j].persistance = 50;
								if( signs[j].count < 0 ) signs[j].count = 0;
								if( signs[j].count > 1000 ) signs[j].count = 1000;
							}
						}
					}
				} /* data */
			} /* comment */
		} /* line */
		close(fd);
	}
	else write_rc();
}

/*** setup sound-interface, determine samplerate and fragment-infos ***/
int setup_sound()
{
	int		fd;
	int		ok;
	int		val;
	audio_buf_info	ainfo;

	fd = open( SOUND_DEVICE, O_RDWR );
	if( fd != -1 )
	{
		ok = 1;
		/* Request 25kSamples/sec. (will be updated by ioctl) */
		SampleRate = 25000;
		if( ioctl( fd, SNDCTL_DSP_SPEED, &SampleRate ) != 0 ) ok = 0;
		if( SampleRate < 4000 ) ok = 0;
		/* Turn to mono */
		val = 0;
		if( ioctl( fd, SNDCTL_DSP_STEREO, &val ) != 0 ) ok = 0;
		/* One channel */
		val = 1;
		if( ioctl( fd, SNDCTL_DSP_CHANNELS, &val ) != 0 ) ok = 0;
		/* Ask for data formats */
		val = 0;
		if( ioctl( fd, SNDCTL_DSP_GETFMTS, &val ) == 0 )
		{
			if( ( val & AFMT_U8 ) != 0 )
			{
				/* Simple 8bit-format supported (enough for this) */
				/*  select this data format */
				val = AFMT_U8;
				if( ioctl( fd, SNDCTL_DSP_SETFMT, &val ) != 0 )	ok = 0;
			}
			else ok = 0;
		}
		else ok = 0;
		/* Ask for output buffers */
		if( ioctl( fd, SNDCTL_DSP_GETOSPACE, &ainfo ) == 0 )
		{
			Fragments = ainfo.fragstotal;
			if( Fragments > WAVEBUFS ) Fragments = WAVEBUFS;
			FragSize = ainfo.fragsize;
			if( FragSize > WAVEBUFLEN ) FragSize = WAVEBUFLEN;
		}
		else ok = 0;
		if( ok == 0 )
		{
			close( fd );
			fd = -1;
		}
	}
	ValuesInBuffer = 0;
	return( fd );
}

/*** allocate space for wave-buffers ***/
int alloc_buffers()
{
	int	i;

	for( i=0; i<Fragments; i++ )
	{
		if( ( WaveBufs[ i ] = (unsigned char *) malloc( FragSize ) ) == NULL ) return 0;
	}
	BufferInUse = 0;
	WaveBuf = WaveBufs[ 0 ];
	ValuesInBuffer = 0;
	return 1;
}

/*** free allocated space for wave-buffers ***/
void free_buffers()
{
	int	i;

	for( i=0; i<Fragments; i++ )
	{
		if( WaveBufs[ i ] == NULL ) return;
		free( WaveBufs[ i ] );
	}
}

/*** determine 'SamplesPerStep' and 'PhiInc' from Freq, SampleRate and WpM ***/
/***   this value is corrected for zero-crossing at end of step using Freq ***/
void calc_step()
{
	int	waves_per_step;
	double	signs_per_second;
	double	steps_per_second;
	double	samples_per_step;
	double	samples_per_wave;

	signs_per_second = ( ( (double ) WpM / 60.0  ) * SIGNS_PER_WORD );
	steps_per_second = signs_per_second * STEPS_PER_SIGN;
	samples_per_step = (double) SampleRate / steps_per_second;
	samples_per_wave = (double) SampleRate / (double) Freq;
	waves_per_step = (int) floor( samples_per_step / samples_per_wave );
	SamplesPerStep = waves_per_step * (int) floor( samples_per_wave );
	PhiInc = ( (double) waves_per_step * TWOPI ) / (double) SamplesPerStep;
}

/*** check for sounddevice ready for new data ***/
int sound_ready()
{
	count_info	cinfo;

	if( SoundDev != -1 ) {
		/* Ask for output counter */
		if( ioctl( SoundDev, SNDCTL_DSP_GETOPTR, &cinfo ) != 0 ) return 0;
		return( ( cinfo.blocks < Fragments - 1 ) ? 1 : 0 );
	}
	else return 0;
}

/*** wait for sounddevice ready for new data ***/
void wait_sound_ready()
{
	struct timespec	wtime;

	wtime.tv_sec = 0;
	wtime.tv_nsec = 10000;	/* in nanoseconds -> 10 ms */
	if( SoundDev != 0 ) {
		while( sound_ready() == 0 ) nanosleep( &wtime, NULL );
	}
}

/*** write wavedata to sound-device ***/
void write_wave()
{
	if( ValuesInBuffer > 0 )
	{
		wait_sound_ready();
		write( SoundDev, WaveBufs[ BufferInUse ], ValuesInBuffer );
		BufferInUse = ( BufferInUse + 1 ) % Fragments;
		WaveBuf = WaveBufs[ BufferInUse ];
		ValuesInBuffer = 0;
	}
}

/*** write wavedata to buffer and then to sound-device ***/
void put_wavedata( unsigned char value )
{
	if( ValuesInBuffer >= FragSize ) {
		write_wave();
	}
	*WaveBuf++ = value;
	ValuesInBuffer++;
}

/*** generate wave-value from angle and amplitude ***/
unsigned char wavefunc( double phi, double ampl )
{
	double	phi2;
	double	fval;

	phi2 = fmod( phi, TWOPI );
	fval = sin( phi2 );
	fval *= ampl;
	fval += 128;
	return( (unsigned char) fval );
}

/*** generate wave ***/
void put_wave( int samples, int ramp )
{
	double	phi;
	double	ampl;
	double	amplinc;
	int	i;

	phi = 0.0;
	if( ramp > 0 )
	{
		amplinc = (double) Ampl / (double) ( ramp + 1 );
		ampl = amplinc;
	} else
	{
		ampl = (double) Ampl;
		amplinc = 0.0;
	}
	for( i=0; i<samples; i++ )
	{
		put_wavedata( wavefunc( phi, ampl ) );
		phi += PhiInc;
		if( i < ramp )
		{
			ampl += amplinc;
		}
		if( i >= samples - ( ramp + 1 ) )
		{
			ampl -= amplinc;
		}
	}
}

/*** generate silence ***/
void put_silence( int samples )
{
	int	i;

	for( i=0; i<samples; i++ ) put_wavedata( 128 );
}

/*** generate cw-sign from code ***/
void put_code( unsigned char * signcode )
{
	int	running;
	int	ramp;

	ramp = (int) floor( ( (double) SamplesPerStep / 100.0 ) * (double) Ramp );
	running = 1;
	while( running )
	{
		switch( *signcode++ )
		{
			case C_DI:
				put_wave( SamplesPerStep, ramp );
				break;
			case C_DA:
				put_wave( SamplesPerStep * RATIO, ramp );
				break;
			default:
				running = 0;
				break;
		}
		if( ( running != 0 ) && ( *signcode != C_END ) ) put_silence( SamplesPerStep );
	}
}

/*** find code for given sign ***/
cwsign * find_sign( char sign )
{
	int	i;

	for(i=0;i<N_SIGNS;i++) if(signs[i].sign == toupper(sign) ) return &signs[i];
	return NULL;
}

/*** generate cw-signs from textstring ***/
void put_text( char * text )
{
	cwsign *	cws;
	double		pause;
	int		ch;

	pause = ( (double) Pause / 100.0 ) + 1.0;
	while( *text != '\0' )
	{
#ifndef OPT_CALIBRATE
		/* note: curses must be initialized to avoid blocking ! */
		ch = getch();
		if( ch == KB_ESC ) break;
#endif
		if( *text == ' ' )
		{
			put_silence( (int) ( (double)( SamplesPerStep * RATIO * RATIO ) * pause ) );
		} else
		{
			if( (cws = find_sign( *text )) != NULL )
			{
				put_code( cws->code );
			}
		}
		text++;
		if( *text != '\0' ) put_silence( (int) ( (double)( SamplesPerStep * RATIO ) * pause ) );
	}
}

/*** generate random sign (using activations and persitance of signs) ***/
int gen_rndsign( char * sign )
{
	int	fails;
	int	index;
	int	persist;
	int	n;
	int	found[RNDMOD];
	double	value[RNDMOD];

	fails = 0;
	*sign = '\0';
	n = 0;
	while( ( fails < 2000 ) && ( n < RNDMOD ) ) {
		index = (int)( ((double)(N_SIGNS - 2)) * (double)rand() / ((double)RAND_MAX + 1.0) );
		if( ( index >= 0 ) && ( index < N_SIGNS - 2 ) ) {
			/* index ok */
			if( ( signs[index].flags & 1 ) != 0 ) {
				/* sign activated */
				persist = (int)( ( 100.0 * (double)rand() ) / ((double)RAND_MAX + 1.0) );
				if( persist <= signs[index].persistance ) {
					/* persistance high enough */
					found[n++] = index;
				}
				else fails++;
			}
			else fails++;
		}
		else fails++;
	}
	if( fails < 2000 ) {
		for( n=0; n<RNDMOD; n++ ) value[n] = (((double)signs[found[n]].persistance) / 20.0 ) / ((((double)signs[found[n]].count) / 50.0 ) + 1.0 );
		index = 0;
		for( n=1; n<RNDMOD; n++ ) if( value[n] > value[index] ) index = n;
		*sign = signs[found[index]].sign;
		signs[index].count++;
		if( signs[index].count >= 1000 ) {
			/* must reduce counters */
			for( index = 0; index < N_SIGNS; index++ ) signs[index].count /= 2;
		}
		return 1;
	}
	return 0;
}

/*** generate groups of 5 signs ***/
int gen_groups( char * text ) 
{
	int	gr;
	int	i;

	*text++ = '>';
	for( gr=0; gr<Groups; gr++ ) {
		*text++ = ' ';
		for( i=0; i<5; i++ ) {
			if( gen_rndsign( text ) == 0 ) return 0;
			text++;
		}
	}
	*text++ = ' ';
	*text++ = '+';
	*text++ = ' ';
	*text = '\0';
	return 1;
}

/********************* Screen and window stuff (curses) ********************/

/* redisplay all visible windows */
void DoRefresh()
{
	wrefresh( Screen );
	touchwin( TopLine );
	wrefresh( TopLine );
	touchwin( TextWnd );
	wrefresh( TextWnd );
	refresh_menu( MainMenu );
	refresh_menu( OptMenu );
	refresh_menu( SigMenu );
	refresh_menu( SigLMenu );
	refresh_menu( LessMenu );
}

/*** check if modification is for all signs ***/
int IsAllKeyMod( VModDat * vmd )
{
	char *	pc;
	
	if( strlen( vmd->text ) > 3 ) {
		pc = vmd->text + strlen( vmd->text ) - 3;
		if( strcmp( pc, "(*)" ) == 0 ) return 1;
	}
	return 0;
}

/*** copy value of signs[0] to all other signs */
void ModAllSigns( VModDat * vmd )
{
	int *		ps;
	void *		pd;
	int		i;
	
	ps = vmd->value;
	pd = (void *) ps + sizeof( cwsign );
	for( i=1; i<N_SIGNS-2; i++ ) {
		*((int *)pd) = *ps;
		pd += sizeof( cwsign );
	}
}

/*** increment a value ***/
int IncValue( void * fdata )
{
	VModDat *	tomod;
	char		numbuf[10];
		
	tomod = (VModDat *) fdata;
	if( *tomod->value < tomod->max ) (*tomod->value)++;
	sprintf( numbuf, tomod->format, *tomod->value );
	wmove( ModVMenu->w, 3, 4 );
	waddstr( ModVMenu->w, numbuf );
	wmove( ModVMenu->w, 0, 0 );
	wrefresh( ModVMenu->w );
	return 1;
}

/* decrement a value ***/
int DecValue( void * fdata )
{
	VModDat *	tomod;
	char		numbuf[10];
		
	tomod = (VModDat *) fdata;
	if( *tomod->value > tomod->min ) (*tomod->value)--;
	sprintf( numbuf, tomod->format, *tomod->value );
	wmove( ModVMenu->w, 3, 4 );
	waddstr( ModVMenu->w, numbuf );
	wmove( ModVMenu->w, 0, 0 );
	wrefresh( ModVMenu->w );
	return 1;
}

/*** quit current menu ***/
int QuitMenu( void * fdata )
{
	return 0;
}

/*** display menu to modify a value ***/
int DoModifyValueMenu( void * fdata )
{
	int		x;
	int		y;
	VModDat *	tomod;
	char		numbuf[10];

	if( fdata != NULL ) {
		tomod = (VModDat *) fdata;
		tomod->sav = *tomod->value;	/* save value */
		ModVMenu = create_menu( 9, 22, tomod->winy, tomod->winx, NULL );
		leaveok( ModVMenu->w, TRUE );
		wbkgdset( ModVMenu->w, A_REVERSE );
		werase( ModVMenu->w );
		x = 2;
		y = 1;
		/* text */
		wmove( ModVMenu->w, y, x );
		waddstr( ModVMenu->w, tomod->text );
		y += 2;
		/* value */
		sprintf( numbuf, tomod->format, *tomod->value );
		wmove( ModVMenu->w, y, x + 2 );
		waddstr( ModVMenu->w, numbuf );
		y += 2;
		/* elements to modify */
		add_menuitem( ModVMenu, x, y, "+", '+', IncValue, fdata );
		add_menuitem( ModVMenu, x + 4, y, "-", '-', DecValue, fdata );
		y += 2;
		add_menuitem( ModVMenu, x, y, "Ok", 'O', NULL, NULL );
#ifdef L_GERMAN
		add_menuitem( ModVMenu, x + 4, y, "Abbruch", 'A', QuitMenu, NULL );
#else
		add_menuitem( ModVMenu, x + 4, y, "Quit", 'Q', QuitMenu, NULL );
#endif
		wmove( MainMenu->w, 0, 0);
		show_menu( ModVMenu );
		if( proc_menu( ModVMenu, 'O' ) == 0 ) {
			/* not terminated using Ok, restore value */
			*tomod->value = tomod->sav;
		}
		else {
			/* new value is ok, copy it to all signs if needed */
			if( IsAllKeyMod( tomod ) ) ModAllSigns( tomod );
		}
		del_menu( ModVMenu );
		ModVMenu = NULL;
		DoRefresh();
	}
	return 1;
}

/*** activate signs for selected lesson ***/
int SetLessSigns( void * fdata )
{
	int		lessidx;
	int		i;
	int		j;
	cwsign *	sig;
		
	lessidx = *(int *)(fdata);
	for( i=0; i<strlen(LessChars); i++ ) {
		sig = find_sign( LessChars[i] );
		if( i < lessidx ) sig->flags |= 1;
		else sig->flags &= ~1;
	} 
	return 0;
}

/*** display the lessons-menu ***/
int DoLessonMenu( void * fdata )
{
	int		x;
	int		y;
	int		ysav;
	int		less;
	int		lidx;
	int 		lessidx[ N_LESSONS ];
	char *		mnutxt;
	char *		pd1;
	char *		pd2;

	mnutxt = malloc( N_LESSONS * 12 );
	pd1 = mnutxt;
	lidx = 0;
	LessMenu = create_menu( 19, 45, 3, 32, NULL );
	leaveok( LessMenu->w, TRUE );
	wbkgdset( LessMenu->w, A_REVERSE );
	werase( LessMenu->w );
	x = 2;
	y = 1;
	/* text */
	wmove( LessMenu->w, y, x + 4 );
	waddstr( LessMenu->w, SMenuText2 );
	y += 2;
	ysav = y;
	for( less=0; less<N_LESSONS; less++ ) {
		/* menu text */
		pd2 = pd1;
		*pd2++ = (char)('A' + less);
		*pd2++ = ' ';
		*pd2++ ='(';
		if( less != 0 ) {
			*pd2++ = '+';
			*pd2++ = ' ';
		}
		*pd2++ = LessChars[ lidx++ ];
		*pd2++ = ' ';
		*pd2++ = LessChars[ lidx++ ];
		if( less == 0 ) {
			*pd2++ = ' ';
			*pd2++ = LessChars[ lidx++ ];
		}
		*pd2++ = ')';
		*pd2++ = '\0';
		lessidx[less] = lidx;
		add_menuitem( LessMenu, x, y, pd1, *pd1, SetLessSigns, (void *) &lessidx[less] );
		pd1 = pd2;
		y += 2;
		if( y > 15 ) {
			y = ysav;
			x += 12;
		}
	}
	/* add leave hint */
	wmove( LessMenu->w, 17, 4 );
	waddstr( LessMenu->w, SMenuText6 );
	show_menu( LessMenu );
/*	wrefresh( LessMenu->w );	*/
	proc_menu( LessMenu, 1 );
	del_menu( LessMenu );
	LessMenu = NULL;
	DoRefresh();
	free(mnutxt);
	return 1;
}

/*** reset counters for all signs ***/
int ResetCounters( void * fdata )
{
	int		i;

	for( i=0; i<N_SIGNS; i++ ) signs[i].count = 0;
	return 0;
}

/*** display a value-list for all signs ***/
void ShowSigListValues( void )
{
	Menuitem *	cmi;
	VModDat *	vmd;
	char		numbuf[10];

	if( SigLMenu != NULL ) {
		cmi = SigLMenu->cur_mi;
		do {
			if( cmi->key != '*' ) {	/* no value for all signs */
				if( ( vmd = (VModDat *) cmi->funcdata ) != NULL ) {
					sprintf( numbuf, vmd->format, *vmd->value );
					wmove( SigLMenu->w, cmi->y, cmi->x + 3 );
					waddstr( SigLMenu->w, numbuf );
				}
			}
			cmi = cmi->next;
		} while( cmi != SigLMenu->cur_mi );
	}
}

/*** display menu with sign-values ***/
int DoSignListMenu( void * fdata )
{
	int		x;
	int		y;
	int		ysav;
	int		i;
	char *		mtext;
	char *		modtxt;
	char *		pd1;
	char *		pd2;
	VModDat *	minfo;
	VModDat 	vmd[N_SIGNS];
	char		menutext[N_SIGNS*2 + 10];
	char		fmt[10];
		
	if( fdata != NULL ) {
		mtext = menutext;
		minfo = (VModDat *) fdata;
		if( minfo->text == SMenuText1 ) {
			strcpy( fmt, "%01d" );
		}
		else {
			strcpy( fmt, "%03d" );
		}
		modtxt = malloc( ( strlen( minfo->text ) + 5 ) * N_SIGNS );
		pd1 = modtxt;
		SigLMenu = create_menu( 19, 45, 3, 32, ShowSigListValues );
		leaveok( SigLMenu->w, TRUE );
		wbkgdset( SigLMenu->w, A_REVERSE );
		werase( SigLMenu->w );
		x = 2;
		y = 1;
		/* text */
		wmove( SigLMenu->w, y, x + 8 );
		waddstr( SigLMenu->w, minfo->text );
		y += 2;
		ysav = y;
		for( i=0; i<N_SIGNS-2; i++ ) {
			/* setup modify-data */
			vmd[i].winx = 5;
			vmd[i].winy = 11;
			strcpy( pd1, minfo->text );
			pd2 = pd1 + strlen( pd1 );
			*pd2++ = ' ';
			*pd2++ = '(';
			*pd2++ = signs[i].sign;
			*pd2++ = ')';
			*pd2++ = '\0';
			vmd[i].text = pd1;
			pd1 = pd2;
			vmd[i].format = fmt;
			if( minfo->text == SMenuText1 ) {
				/* Activation selected, use flags element of signs */
				vmd[i].value = &signs[i].flags;
				vmd[i].min = 0;
				vmd[i].max = 1;
			}
			else {
				/* Persistance selected, use persistance element */
				vmd[i].value = &signs[i].persistance;
				vmd[i].min = 1;
				vmd[i].max = 100;
			}
/*			vmd[i].sav = *(vmd[i].value);  */	/* No need to save value here */
			/* menu-entry */
			*mtext++ = signs[i].sign;
			*mtext++ = '\0';
			add_menuitem( SigLMenu, x, y, mtext - 2, signs[i].sign, DoModifyValueMenu, (void *) &vmd[i] );
			y += 2;
			if( y > 15 ) {
				y = ysav;
				x += 7;
			}
		}
		/* add entry for all signs */
		i = N_SIGNS-2;
		vmd[i].winx = 5;
		vmd[i].winy = 11;
		strcpy( pd1, minfo->text );
		pd2 = pd1 + strlen( pd1 );
		strcpy( pd2, " (*)" );
		vmd[i].text = pd1;
		vmd[i].format = fmt;
		if( minfo->text == SMenuText1 ) {
			/* Activation selected, use flags element of signs */
			vmd[i].value = &signs[0].flags;
			vmd[i].min = 0;
			vmd[i].max = 1;
		}
		else {
			/* Persistance selected, use persistance element */
			vmd[i].value = &signs[0].persistance;
			vmd[i].min = 0;
			vmd[i].max = 100;
		}
		*mtext++ = '*';
		*mtext++ = '\0';
		add_menuitem( SigLMenu, x, y, mtext - 2, '*', DoModifyValueMenu, (void *) &vmd[i] );
		/* add leave hint */
		wmove( SigLMenu->w, 17, 4 );
		waddstr( SigLMenu->w, SMenuText6 );
		show_menu( SigLMenu );
		ShowSigListValues();		/* fillin values */
		wrefresh( SigLMenu->w );
		proc_menu( SigLMenu, 1 );
		del_menu( SigLMenu );
		SigLMenu = NULL;
		DoRefresh();
		free(modtxt);
	}
	return 1;
}

/*** display menu to select the kind of value for signs to modify ***/
int DoSignsMenu( void * fdata )
{
	int		x;
	int		y;
	VModDat 	vmd[2];
	int		sigindex;	
	
	vmd[0].text = SMenuText1;
	vmd[0].value = &sigindex;
	vmd[0].max = N_SIGNS - 1;
	vmd[0].min = 0;
	vmd[0].sav = 0;
	vmd[1].text = SMenuText3;
	vmd[1].value = &sigindex;
	vmd[1].max = N_SIGNS - 3;	/* last 2 signs ( ka + ar ) are not selectable */
	vmd[1].min = 0;
	vmd[1].sav = 0;
	SigMenu = create_menu( 8, 20, 2, 10, NULL );
	leaveok( SigMenu->w, TRUE );
	wbkgdset( SigMenu->w, A_REVERSE );
	werase( SigMenu->w );
	x = 2;
	y = 1;
	add_menuitem( SigMenu, x, y, SMenuText1, SMENU_KEY1, DoSignListMenu, (void *) &vmd[0] );
	y++;
	add_menuitem( SigMenu, x, y, SMenuText2, SMENU_KEY2, DoLessonMenu, NULL );
	y++;
	add_menuitem( SigMenu, x, y, SMenuText3, SMENU_KEY3, DoSignListMenu, (void *) &vmd[1] );
	y++;
	add_menuitem( SigMenu, x, y, SMenuText4, SMENU_KEY4, ResetCounters, NULL );
	y +=2;
	add_menuitem( SigMenu, x, y, SMENU_TEXT5, SMENU_KEY5, NULL, NULL );
	show_menu( SigMenu );
	proc_menu( SigMenu, SMENU_KEY5 );
	del_menu( SigMenu );
	SigMenu = NULL;
	DoRefresh();
	return 1;
}

/*** display options-menu ***/
int DoOptionsMenu( void * fdata )
{
	int		x;
	int		y;
	VModDat 	vmd[6];

	vmd[0].winx = 20;
	vmd[0].winy = 4;
	vmd[0].text = OMenuText1;
	vmd[0].value = &Freq;
	vmd[0].max = MAX_FRQ;
	vmd[0].min = MIN_FRQ;
	vmd[0].sav = Freq;
	vmd[0].format = "%04d";	
	vmd[1].winx = 20;
	vmd[1].winy = 4;
	vmd[1].text = OMenuText2;
	vmd[1].value = &Ampl;
	vmd[1].max = MAX_AMP;
	vmd[1].min = MIN_AMP;
	vmd[1].sav = Ampl;	
	vmd[1].format = "%03d";
	vmd[2].winx = 20;
	vmd[2].winy = 4;
	vmd[2].text = OMenuText3;
	vmd[2].value = &WpM;
	vmd[2].max = MAX_WPM;
	vmd[2].min = MIN_WPM;
	vmd[2].sav = WpM;
	vmd[2].format = "%02d";	
	vmd[3].winx = 20;
	vmd[3].winy = 4;
	vmd[3].text = OMenuText4;
	vmd[3].value = &Pause;
	vmd[3].max = MAX_PSE;
	vmd[3].min = MIN_PSE;
	vmd[3].sav = Pause;
	vmd[3].format = "%03d";
	vmd[4].winx = 20;
	vmd[4].winy = 4;
	vmd[4].text = OMenuText5;
	vmd[4].value = &Ramp;
	vmd[4].max = MAX_KEY;
	vmd[4].min = MIN_KEY;
	vmd[4].sav = Ramp;
	vmd[4].format = "%02d";
	vmd[5].winx = 20;
	vmd[5].winy = 4;
	vmd[5].text = OMenuText6;
	vmd[5].value = &Groups;
	vmd[5].max = MAX_GRP;
	vmd[5].min = MIN_GRP;
	vmd[5].sav = Groups;
	vmd[5].format = "%02d";
	OptMenu = create_menu( 10, 20, 2, 0, NULL );
	leaveok( OptMenu->w, TRUE );
	wbkgdset( OptMenu->w, A_REVERSE );
	werase( OptMenu->w );
	x = 2;
	y = 1;
	add_menuitem( OptMenu, x, y, OMenuText1, OMENU_KEY1, DoModifyValueMenu, (void *) &vmd[0] );
	y++;
	add_menuitem( OptMenu, x, y, OMenuText2, OMENU_KEY2, DoModifyValueMenu, (void *) &vmd[1] );
	y++;
	add_menuitem( OptMenu, x, y, OMenuText3, OMENU_KEY3, DoModifyValueMenu, (void *) &vmd[2] );
	y++;
	add_menuitem( OptMenu, x, y, OMenuText4, OMENU_KEY4, DoModifyValueMenu, (void *) &vmd[3] );
	y++;
	add_menuitem( OptMenu, x, y, OMenuText5, OMENU_KEY5, DoModifyValueMenu, (void *) &vmd[4] );
	y++;
	add_menuitem( OptMenu, x, y, OMenuText6, OMENU_KEY6, DoModifyValueMenu, (void *) &vmd[5] );
	y +=2;
	add_menuitem( OptMenu, x, y, OMENU_TEXT7, OMENU_KEY7, NULL, NULL );
	show_menu( OptMenu );
	proc_menu( OptMenu, OMENU_KEY7 );
	del_menu( OptMenu );
	OptMenu = NULL;
	DoRefresh();
	calc_step();
	return 1;
}

/*** create top line ***/
void OpenTopLine( void )
{
	char	buffer[100];
	int	x;

	TopLine = newwin( 1, Screen->_maxx + 1, 0, 0 );
	leaveok( TopLine, TRUE );
	werase( TopLine );
	x = ( TopLine->_maxx - strlen( ApplText ) ) / 2;
	wmove( TopLine, 0, x );
	waddstr( TopLine, ApplText );
	wrefresh( TopLine );
}

/*** remove top line ***/
void CloseTopLine( void )
{
	if( TopLine != NULL ) {
		delwin( TopLine );
		TopLine = NULL;
	}	
}

/*** create text screen ***/
void OpenTextWnd( void )
{
	char	buffer[100];
	int	x;

	TextWnd = newwin( Screen->_maxy - 1, Screen->_maxx + 1, 2, 0 );
	leaveok( TextWnd, TRUE );
	scrollok( TextWnd, TRUE );
	werase( TextWnd );
	wmove( TextWnd, 1, 0 );
	wrefresh( TextWnd );
}

/*** remove text screen ***/
void CloseTextWnd( void )
{
	if( TextWnd != NULL ) {
		delwin( TextWnd );
		TextWnd = NULL;
	}	
}

/*** Play a couple of random groups ***/
int PlaySigns( void * fdata )
{
	char *	signbuf;
	char *	ps;
	
	signbuf = malloc( Groups * 6 + 10 );
	if( signbuf != NULL ) {
		if( gen_groups( signbuf ) ) {		
			put_text( signbuf );
			write_wave();		// output buffered data
			ps = signbuf;
			while( *ps != '\0' ) *ps++ = tolower( *ps );
			waddstr( TextWnd, signbuf );
			waddch( TextWnd, '\n' );
		}
		else waddstr( TextWnd, "Sorry, cannot create CW-Signs\n" );
		if( TextWnd->_cury > TextWnd->_maxy - 1 ) {
			scroll( TextWnd );
			wmove( TextWnd, TextWnd->_maxy - 1, 0 );
		}
		wrefresh( TextWnd );
		free( signbuf );
	}
	return 1;	
}

/*** create main menu ***/
void OpenMainMenu( void )
{
	int		x;
	int		y;

	MainMenu = create_menu( 1, Screen->_maxx + 1, 1, 0, NULL );
	leaveok( MainMenu->w, TRUE );
	wbkgdset( MainMenu->w, A_REVERSE );
	werase( MainMenu->w );
	x = 2;
	y = 0;
	add_menuitem( MainMenu, x, y, MMENU_TEXT1, MMENU_KEY1, DoOptionsMenu, NULL );
	x += strlen( MMENU_TEXT1 ) + 3;
	add_menuitem( MainMenu, x, y, MMENU_TEXT2, MMENU_KEY2, DoSignsMenu, NULL );
	x += strlen( MMENU_TEXT2 ) + 3;
	add_menuitem( MainMenu, x, y, MMENU_TEXT3, MMENU_KEY3, PlaySigns, NULL );
	x = MainMenu->w->_maxx - strlen( MMENU_TEXT4 ) - 3;
	add_menuitem( MainMenu, x, y, MMENU_TEXT4, MMENU_KEY4, NULL, NULL );
	show_menu( MainMenu );
}

/*** remove main menu ***/
void CloseMainMenu( void )
{
	if( MainMenu != NULL ) {
		del_menu( MainMenu );
		MainMenu = NULL;
	}
}

/*** create root-window ***/
int OpenMainScreen( void )
{
	char	buffer[100];
	int	x;

	Screen = initscr();
	if( ( Screen->_maxx + 1 >= SCR_MINX ) && ( Screen->_maxy + 1 >= SCR_MINY ) ) {
		noecho();
		cbreak();
		nodelay( Screen, TRUE );
		werase( Screen );
		wrefresh( Screen );
		/* create top-line */
		OpenTopLine();
		/* create main menu */
		OpenMainMenu();
		/* create text window */
		OpenTextWnd();
		wmove( TopLine, 0, 0 );
		wrefresh( TopLine );
		proc_menu( MainMenu, MMENU_KEY4 );
	}
	else {
		endwin();
		printf("\n Sorry, program needs %d x %d (or more) char screen,\n", SCR_MINX, SCR_MINY);
		printf("  only %d x %d chars are available\n", Screen->_maxx +1, Screen->_maxy +1 );
		return 0;
	}
	return 1;
}

/*** go back to normal mode ***/
int CloseMainScreen( void )
{
	CloseTextWnd();
	CloseTopLine();
	CloseMainMenu();
	echo();
	endwin();
	return 1;
}

/******************************************************************************/
/* Main program                                                               */
/******************************************************************************/
int main( int argc, char * argv[] )
{
	int		i;
	int		temp;
	time_t		t1;
	time_t		t2;

	if( ( SoundDev = setup_sound() ) == -1 )
	{
		printf("\n Sorry, cannot access/use the sound-device\n");
		return 0;
	}
	/* allocate space for wave-buffers */
	if( alloc_buffers() == 0 )
	{
		free_buffers();
		printf("\n Sorry, cannot allocate memory for wave-buffers\n");
		return 0;
	}
	/* Determine cw-paras (depends on sound-device) */
	calc_step();
	/* Read (or generate) rc-file */
	read_rc();

#ifdef	OPT_CALIBRATE
/*  ---- only used for calibration of SIGNS_PER_WORD and STEPS_PER_SIGN ---*/
	temp = Pause;
	Pause = 0;	/* no additional pause while calibrating ! */
	time( &t1 );
	for(i=0;i<WpM;i++)
	{
		put_text("paris ");
	}
	time( &t2 );
	printf("\n time: %d \n", t2 - t1 );	/* should output 60 seconds */
	Pause = temp;
#else	/* OPT_CALIBRATE */
/*  ---- normal operation ---*/
	/* initialize random generator */
	srand( time( &t1 ) );
	calc_step();
	/* setup screen amd keyboard using curses */
	if( OpenMainScreen() ) {	/* most of program done here */
		calc_step();
		put_text("73");
		/* set screen and keyboard back to normal mode */
		CloseMainScreen();
	}
#endif	/* OPT_CALIBRATE */
/*	put_silence( FragSize );	*/	/* hmm, needed to output rest ???? */
	write_wave();					/* output buffered data	*/
	close( SoundDev );
	free_buffers();
	write_rc();
	printf("\n");
	return 0;
}

